<?php

use Transbank\WooCommerce\Webpay\Controllers\ResponseController;
use Transbank\WooCommerce\Webpay\Controllers\ThanksPageController;
use Transbank\WooCommerce\Webpay\Exceptions\TokenNotFoundOnDatabaseException;
use Transbank\WooCommerce\Webpay\Helpers\RedirectorHelper;
use Transbank\WooCommerce\Webpay\TransbankWebpayOrders;

if (!defined('ABSPATH')) {
    exit();
} // Exit if accessed directly

/**
 * Plugin Name: Webpay Plus Webservices de Transbank para WooCommerce
 * Plugin URI: https://andres.reyes.dev
 * Description: Recibe pagos en línea con Tarjetas de Crédito y Redcompra en tu WooCommerce a través de Webpay Plus.
 * Version: 20.5.9
 * Author: Andrés Reyes Galgani
 * Author URI: https://andres.reyes.dev
 *
 * Requires at least: 4.0
 * Tested up to: 5.5
 * Requires PHP: 5.6+
 *
 * WC requires at least: 2.5
 * WC tested up to: 4.2
 */

add_action('plugins_loaded', 'woocommerce_transbank_init', 0);

require_once plugin_dir_path(__FILE__) . "vendor/autoload.php";
require_once plugin_dir_path(__FILE__) . "libwebpay/TransbankSdkWebpay.php";

register_activation_hook(__FILE__, 'on_webpay_plugin_activation');
add_action('admin_init', 'on_transbank_webpay_plugins_loaded');
add_filter('woocommerce_payment_gateways', 'woocommerce_add_transbank_gateway');

add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'add_action_links');

function woocommerce_transbank_init()
{
    if (!class_exists("WC_Payment_Gateway")) {
        return;
    }

    class WC_Gateway_Transbank extends WC_Payment_Gateway
    {
        private static $URL_RETURN;
        private static $URL_FINAL;

        var $notify_url;
        var $plugin_url;

        public function __construct()
        {

            self::$URL_RETURN = home_url('/') . '?wc-api=WC_Gateway_transbank';
            self::$URL_FINAL = '_URL_';

            $this->id = 'transbank';
            $this->icon = "https://payment.swo.cl/host/logo";
            $this->method_title = __('Webpay Plus Webservices');
            $this->method_description = __('Paga con las Tarjetas de Crédito y Redcompra de Transbank');
            $this->notify_url = add_query_arg('wc-api', 'WC_Gateway_' . $this->id, home_url('/'));
            $this->title = 'Tarjetas de Crédito o Redcompra — Webpay Plus de Transbank';
            $this->description = 'Permite el pago de productos y/o servicios, con tarjetas de cr&eacute;dito y Redcompra a trav&eacute;s de Webpay Plus';
            $this->plugin_url = plugins_url('/', __FILE__);


            $certificates = include 'libwebpay/certificates.php';
            $webpay_commerce_code = $certificates['commerce_code'];
            $webpay_private_key = $certificates['private_key'];
            $webpay_public_cert = $certificates['public_cert'];
            $webpay_webpay_cert = (new TransbankSdkWebpay(null))->getWebPayCertDefault();

            $this->config = [
                "MODO" => trim($this->get_option('webpay_test_mode', 'INTEGRACION')),
                "COMMERCE_CODE" => trim($this->get_option('webpay_commerce_code', $webpay_commerce_code)),
                "PRIVATE_KEY" => trim(str_replace("<br/>", "\n",
                    $this->get_option('webpay_private_key', $webpay_private_key))),
                "PUBLIC_CERT" => trim(str_replace("<br/>", "\n",
                    $this->get_option('webpay_public_cert', $webpay_public_cert))),
                "WEBPAY_CERT" => trim(str_replace("<br/>", "\n",
                    $this->get_option('webpay_webpay_cert', $webpay_webpay_cert))),
                "URL_RETURN" => home_url('/') . '?wc-api=WC_Gateway_' . $this->id,
                "URL_FINAL" => "_URL_",
                "ECOMMERCE" => 'woocommerce',
                "VENTA_DESC" => [
                    "VD" => "Venta Deb&iacute;to",
                    "VN" => "Venta Normal",
                    "VC" => "Venta en cuotas",
                    "SI" => "3 cuotas sin inter&eacute;s",
                    "S2" => "2 cuotas sin inter&eacute;s",
                    "NC" => "N cuotas sin inter&eacute;s"
                ],
                "STATUS_AFTER_PAYMENT" => $this->get_option('after_payment_order_status')
            ];


            /**
             * Carga configuración y variables de inicio
             **/

            $this->init_form_fields();
            $this->init_settings();

            add_action('woocommerce_thankyou', [new ThanksPageController($this->config), 'show'], 1);
            add_action('woocommerce_receipt_' . $this->id, [$this, 'receipt_page']);
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
            add_action('woocommerce_api_wc_gateway_' . $this->id, [$this, 'check_ipn_response']);

            if (!$this->is_valid_for_use()) {
                $this->enabled = false;
            }
        }

        /**
         * Comprueba configuración de moneda (Peso Chileno)
         **/
        function is_valid_for_use()
        {
            if (!in_array(get_woocommerce_currency(),
                apply_filters('woocommerce_' . $this->id . '_supported_currencies', ['CLP']))) {
                return false;
            }

            return true;
        }

        /**
         * Inicializar campos de formulario
         **/
        function init_form_fields()
        {
            $this->form_fields = [
                'enabled' => [
                    'title' => __('Activar/Desactivar', 'woocommerce'),
                    'type' => 'checkbox',
                    'default' => 'yes'
                ],
                'webpay_test_mode' => [
                    'title' => __('Ambiente', 'woocommerce'),
                    'type' => 'select',
                    'options' => [
                        'INTEGRACION' => 'Integraci&oacute;n',
                        'PRODUCCION' => 'Producci&oacute;n'
                    ],
                    'default' => 'INTEGRACION'
                ],
                'webpay_commerce_code' => [
                    'title' => __('C&oacute;digo de Comercio', 'woocommerce'),
                    'type' => 'text',
                    'default' => __($this->config['COMMERCE_CODE'], 'woocommerce')
                ],
                'webpay_private_key' => [
                    'title' => __('Llave Privada', 'woocommerce'),
                    'type' => 'textarea',
                    'default' => __(str_replace("<br/>", "\n", $this->config['PRIVATE_KEY']), 'woocommerce'),
                    'css' => 'font-family: monospace'
                ],
                'webpay_public_cert' => [
                    'title' => __('Certificado', 'woocommerce'),
                    'type' => 'textarea',
                    'default' => __(str_replace("<br/>", "\n", $this->config['PUBLIC_CERT']), 'woocommerce'),
                    'css' => 'font-family: monospace'
                ],
                'after_payment_order_status' => [
                    'title' => __('Estado de pedido después del pago'),
                    'type' => 'select',
                    'options' => [
                        'wc-pending' => 'Pendiente',
                        'wc-processing' => 'Procesando',
                        'wc-on-hold' => 'Retenido',
                        'wc-completed' => 'Completado',
                        'wc-cancelled' => 'Cancelado',
                        'wc-refunded' => 'Reembolsado',
                        'wc-failed' => 'Fallido'
                    ],
                    'default' => 'wc-processing'
                ]
            ];
        }

        /**
         * Pagina Receptora
         **/
        function receipt_page($order_id)
        {
            $order = new WC_Order($order_id);
            $amount = (int)number_format($order->get_total(), 0, ',', '');
            $sessionId = uniqid();
            $buyOrder = $order_id;
            $returnUrl = self::$URL_RETURN;
            $finalUrl = str_replace("_URL_",
                add_query_arg('key', $order->get_order_key(), $order->get_checkout_order_received_url()),
                self::$URL_FINAL);

            $transbankSdkWebpay = new TransbankSdkWebpay($this->config);
            $result = $transbankSdkWebpay->initTransaction($amount, $sessionId, $buyOrder, $returnUrl, $finalUrl);

            if (!isset($result["token_ws"])) {
                wc_add_notice(__('ERROR: ',
                        'woocommerce') . 'Ocurrió un error al intentar conectar con WebPay Plus. Por favor intenta mas tarde.<br/>',
                    'error');
                return;
            }

            $url = $result["url"];
            $token_ws = $result["token_ws"];

            TransbankWebpayOrders::createTransaction([
                'order_id' => $order_id,
                'buy_order' => $buyOrder,
                'amount' => $amount,
                'token' => $token_ws,
                'session_id' => $sessionId,
                'status' => TransbankWebpayOrders::STATUS_INITIALIZED
            ]);

            RedirectorHelper::redirect($url, ["token_ws" => $token_ws]);
        }

        /**
         * Obtiene respuesta IPN (Instant Payment Notification)
         **/
        function check_ipn_response()
        {
            @ob_clean();
            if (isset($_POST)) {
                header('HTTP/1.1 200 OK');
                return (new ResponseController($this->config))->response($_POST);
            } else {
                echo "Ocurrio un error al procesar su compra";
            }
        }

        /**
         * Procesar pago y retornar resultado
         **/
        function process_payment($order_id)
        {
            $order = new WC_Order($order_id);

            return [
                'result' => 'success',
                'redirect' => $order->get_checkout_payment_url(true)
            ];
        }


    }

    /**
     * Añadir Transbank Plus a Woocommerce
     **/
    function woocommerce_add_transbank_gateway($methods)
    {
        $methods[] = 'WC_Gateway_transbank';

        return $methods;
    }

    /**
     * Muestra detalle de pago a Cliente a finalizar compra
     **/
    function pay_transbank_webpay_content($orderId)
    {

    }


}

function wc_transbank_message($hook_suffix) {
    $current_section = (isset($_GET['section']) && !empty($_GET['section']))? $_GET['section']:'';
    if($current_section == 'transbank') {
        $bg = plugin_dir_url( __FILE__ ) . 'assets/img/back.png';
        $logo = plugin_dir_url( __FILE__ ) . 'assets/img/logo_webpay.png';
        $name = get_currentuserinfo()->user_firstname . ' ' . get_currentuserinfo()->user_lastname;
        $domain = get_home_url();

        $message = <<<EOL
<div class="postbox" style="background-image: url('$bg'); border-radius: 15px">
<div class="inside" style="text-align: center; color: #ffffff; font-weight: 300; padding: 10px;">
<p style="font-size: 25px; font-weight: 300;">¿Necesitas ayuda con la Integración?</p>
<p style="font-size: 17px; font-weight: 300;">Soy <a href="https://andres.reyes.dev" style="color: white"><strong>Andrés Reyes Galgani</strong></a>, principal integrador de Transbank y estoy aquí para ayudarte.</p>
<p style="font-size: 17px; font-weight: 300;">Te acompaño en todo el proceso de contratación, asesoria, evaluación del servidor, integración y paso a producción.<br />Todo para que tu comercio venda en línea en menos de 5 días hábiles.</p>
<p style="font-size: 17px; font-weight: 300;">Evita problemas con las ventas u órdenes que no llegaron. Mi servicio es integral y garantizado.</p>
<p style="font-size: 17px; font-weight: 300;">Si bien puedes usar mi plugin sin costo, siempre es conveniente asesorarse por un profesional. Y estoy aquí, a un click.</p>
<hr style="max-width: 50%; border: 1px solid #ffffff; padding: 0 15px"/>
<p style="font-size: 15px; font-weight: 300;">El costo del servicio indicado es de $90.000 <small>líquido</small>. Sin cobros sorpresa adicionales.</p>
<hr style="max-width: 50%; border: 1px solid #ffffff; padding: 0 15px"/>
<p style="text-align: center;"><a class="button" style="padding: 10px 20px; margin-top: 20px; margin-bottom: 15px; background-color: #11a94a; color: white; font-size: 15px; font-weight: 300;" href="https://link.reyes.dev/webpay-plus-woocommerce?text=Hola, mi nombre es $name y necesito integrar Webpay Plus en el dominio $domain" target="_blank">¡Háblame Ahora por WhatsApp!</a></p>
<p><img src="$logo" style="border-radius: 100px; max-width: 100px; margin-bottom: 15px; box-shadow: 0 0 15px 15px #e28dbc;" /></p>
</div>
</div>
EOL;
        echo $message;
    }
}
add_action('woocommerce_settings_checkout', 'wc_transbank_message');

function add_action_links($links)
{
    $newLinks = [
        '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=transbank') . '">💳 Configuración de Webpay Plus</a>',
    ];

    return array_merge($links, $newLinks);
}

function on_webpay_plugin_activation()
{
    woocommerce_transbank_init();
}

function on_transbank_webpay_plugins_loaded()
{
    TransbankWebpayOrders::createTableIfNeeded();
}

function transbank_remove_database()
{
    TransbankWebpayOrders::deleteTable();
}

register_uninstall_hook(__FILE__, 'transbank_remove_database');
